#property copyright "Copyright 2012, Rone. Enhanced 2026"
#property link      "rone.sergey@gmail.com"
#property version   "2.02"
#property description "Automatic trend lines - 3 sets (MT4 Version)"
#property indicator_chart_window
#property indicator_buffers 0
#property strict
//---
enum ENUM_LINE_TYPE {
   EXM_EXM,    // 1: By 2 extremums
   EXM_DELTA   // 2: Extremum and delta
};
//+------------------------------------------------------------------+
//| input parameters                                                 |
//+------------------------------------------------------------------+
input ENUM_LINE_TYPE InpLineType = EXM_DELTA;// Line type

// Set 1 - Original
input int            InpLeftExmSide1 = 20;    // Set1: Left extremum side
input int            InpRightExmSide1 = 3;    // Set1: Right extremum side
input int            InpFromCurrent1 = 3;     // Set1: Offset from current bar

// Set 2 - Medium
input int            InpLeftExmSide2 = 50;    // Set2: Left extremum side
input int            InpRightExmSide2 = 3;    // Set2: Right extremum side
input int            InpFromCurrent2 = 3;     // Set2: Offset from current bar

// Set 3 - Long
input int            InpLeftExmSide3 = 100;   // Set3: Left extremum side
input int            InpRightExmSide3 = 6;    // Set3: Right extremum side
input int            InpFromCurrent3 = 6;     // Set3: Offset from current bar

input bool           InpPrevExmBar = false;   // Account for bar before extremum
input int            InpMaxLookback = 5000;   // Maximum bars to look back
input double         InpAlertOffset = 0.01;    // Alert offset (% of price)

//---
input int            InpLinesWidth = 1;       // Lines width
input color          InpSupColor1 = clrRed;   // Set1: Support color
input color          InpResColor1 = clrBlue;  // Set1: Resistance color
input color          InpSupColor2 = clrOrangeRed;   // Set2: Support color
input color          InpResColor2 = clrDodgerBlue;  // Set2: Resistance color
input color          InpSupColor3 = clrDarkRed;     // Set3: Support color
input color          InpResColor3 = clrNavy;        // Set3: Resistance color

//--- Button position (distance from RIGHT-BOTTOM corner)
input int            InpButtonXOffset = 140;  // Distance from right edge
input int            InpButtonYOffset = 50;   // Distance from bottom edge
input int            InpButtonSpacing = 40;   // Space between buttons

//--- global variables
int            minRequiredBars[3];
bool           showLines = false;
bool           alertsEnabled = false;
datetime       lastAlertTime = 0;
datetime       lastCalcTime = 0;

// Store line points
double         leftSupPrice[3], rightSupPrice[3], leftResPrice[3], rightResPrice[3];
datetime       leftSupTime[3], rightSupTime[3], leftResTime[3], rightResTime[3];

//--- Button names
string btnShowLines = "btnShowLines";
string btnAlerts = "btnAlerts";

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit() {
   minRequiredBars[0] = InpLeftExmSide1 * 2 + MathMax(InpRightExmSide1, InpFromCurrent1) * 2;
   minRequiredBars[1] = InpLeftExmSide2 * 2 + MathMax(InpRightExmSide2, InpFromCurrent2) * 2;
   minRequiredBars[2] = InpLeftExmSide3 * 2 + MathMax(InpRightExmSide3, InpFromCurrent3) * 2;
   
   // Initialize arrays
   ArrayInitialize(leftSupPrice, 0);
   ArrayInitialize(rightSupPrice, 0);
   ArrayInitialize(leftResPrice, 0);
   ArrayInitialize(rightResPrice, 0);
   ArrayInitialize(leftSupTime, 0);
   ArrayInitialize(rightSupTime, 0);
   ArrayInitialize(leftResTime, 0);
   ArrayInitialize(rightResTime, 0);
   
   // Create control buttons
   CreateButton(btnShowLines, InpButtonXOffset, InpButtonYOffset + InpButtonSpacing, 120, 30, "Lines: OFF", clrWhite, clrRed);
   CreateButton(btnAlerts, InpButtonXOffset, InpButtonYOffset, 120, 30, "Alerts: OFF", clrWhite, clrRed);
   
   return(INIT_SUCCEEDED);
}
//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
   for(int i = 1; i <= 3; i++) {
      ObjectDelete("Current_Support_" + IntegerToString(i));
      ObjectDelete("Current_Resistance_" + IntegerToString(i));
   }
   ObjectDelete(btnShowLines);
   ObjectDelete(btnAlerts);
}
//+------------------------------------------------------------------+
//| ChartEvent function                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam) {
   if(id == CHARTEVENT_OBJECT_CLICK) {
      if(sparam == btnShowLines) {
         showLines = !showLines;
         ObjectSetText(btnShowLines, showLines ? "Lines: ON" : "Lines: OFF");
         ObjectSet(btnShowLines, OBJPROP_BGCOLOR, showLines ? clrGreen : clrRed);
         
         for(int i = 1; i <= 3; i++) {
            if(showLines) {
               ObjectSet("Current_Support_" + IntegerToString(i), OBJPROP_TIMEFRAMES, OBJ_ALL_PERIODS);
               ObjectSet("Current_Resistance_" + IntegerToString(i), OBJPROP_TIMEFRAMES, OBJ_ALL_PERIODS);
            } else {
               ObjectSet("Current_Support_" + IntegerToString(i), OBJPROP_TIMEFRAMES, OBJ_NO_PERIODS);
               ObjectSet("Current_Resistance_" + IntegerToString(i), OBJPROP_TIMEFRAMES, OBJ_NO_PERIODS);
            }
         }
         ObjectSet(btnShowLines, OBJPROP_STATE, false);
         ChartRedraw();
      }
      else if(sparam == btnAlerts) {
         alertsEnabled = !alertsEnabled;
         ObjectSetText(btnAlerts, alertsEnabled ? "Alerts: ON" : "Alerts: OFF");
         ObjectSet(btnAlerts, OBJPROP_BGCOLOR, alertsEnabled ? clrGreen : clrRed);
         ObjectSet(btnAlerts, OBJPROP_STATE, false);
         ChartRedraw();
      }
   }
}
//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int start() {
   int bars = Bars;
   int leftSides[3], rightSides[3], fromCurrents[3];
   
   leftSides[0] = InpLeftExmSide1;
   rightSides[0] = InpRightExmSide1;
   fromCurrents[0] = InpFromCurrent1;
   
   leftSides[1] = InpLeftExmSide2;
   rightSides[1] = InpRightExmSide2;
   fromCurrents[1] = InpFromCurrent2;
   
   leftSides[2] = InpLeftExmSide3;
   rightSides[2] = InpRightExmSide3;
   fromCurrents[2] = InpFromCurrent3;

   // Only recalculate on new bar
   if(Time[0] != lastCalcTime) {
      lastCalcTime = Time[0];
      
      for(int set = 0; set < 3; set++) {
         if(bars < minRequiredBars[set]) continue;
         
         int leftIndex, rightIndex;
         double delta, tmpDelta;
         int maxLeftIndex = MathMin(bars - 1, InpMaxLookback);
         
         switch(InpLineType) {
            case EXM_DELTA: {
               //--- Support Left Point (find lowest low)
               leftIndex = leftSides[set];
               for(; leftIndex < maxLeftIndex && !isLowestLow(leftIndex, leftSides[set]); leftIndex++);
               leftSupPrice[set] = Low[leftIndex];
               leftSupTime[set] = Time[leftIndex];
               
               //--- Support Right Point
               rightIndex = fromCurrents[set];
               delta = (Low[rightIndex] - Low[leftIndex]) / (leftIndex - rightIndex);
               int tmpLeftIndex = InpPrevExmBar ? leftIndex : leftIndex - 1;
               for(int tmpIndex = rightIndex + 1; tmpIndex < tmpLeftIndex; tmpIndex++) {
                  tmpDelta = (Low[tmpIndex] - leftSupPrice[set]) / (leftIndex - tmpIndex);
                  if(tmpDelta < delta) {
                     delta = tmpDelta;
                     rightIndex = tmpIndex;
                  }
               }
               rightSupPrice[set] = Low[rightIndex];
               rightSupTime[set] = Time[rightIndex];

               //--- Resistance Left Point (find highest high)
               leftIndex = leftSides[set];
               for(; leftIndex < maxLeftIndex && !isHighestHigh(leftIndex, leftSides[set]); leftIndex++);
               leftResPrice[set] = High[leftIndex];
               leftResTime[set] = Time[leftIndex];
               
               //--- Resistance Right Point
               rightIndex = fromCurrents[set];
               delta = (High[leftIndex] - High[rightIndex]) / (leftIndex - rightIndex);
               tmpLeftIndex = InpPrevExmBar ? leftIndex : leftIndex - 1;
               for(int tmpIndex = rightIndex + 1; tmpIndex < tmpLeftIndex; tmpIndex++) {
                  tmpDelta = (leftResPrice[set] - High[tmpIndex]) / (leftIndex - tmpIndex);
                  if(tmpDelta < delta) {
                     delta = tmpDelta;
                     rightIndex = tmpIndex;
                  }
               }
               rightResPrice[set] = High[rightIndex];
               rightResTime[set] = Time[rightIndex];
            }
            break;
               
            case EXM_EXM:
            default: {
               //--- Support Right Point
               rightIndex = rightSides[set];
               for(; rightIndex < maxLeftIndex && !isLowestLow(rightIndex, rightSides[set]); rightIndex++);
               rightSupPrice[set] = Low[rightIndex];
               rightSupTime[set] = Time[rightIndex];
               
               //--- Support Left Point
               leftIndex = rightIndex + rightSides[set];
               for(; leftIndex < maxLeftIndex && !isLowestLow(leftIndex, leftSides[set]); leftIndex++);
               leftSupPrice[set] = Low[leftIndex];
               leftSupTime[set] = Time[leftIndex];

               //--- Resistance Right Point
               rightIndex = rightSides[set];
               for(; rightIndex < maxLeftIndex && !isHighestHigh(rightIndex, rightSides[set]); rightIndex++);
               rightResPrice[set] = High[rightIndex];
               rightResTime[set] = Time[rightIndex];
               
               //--- Resistance Left Point
               leftIndex = rightIndex + rightSides[set];
               for(; leftIndex < maxLeftIndex && !isHighestHigh(leftIndex, leftSides[set]); leftIndex++);
               leftResPrice[set] = High[leftIndex];
               leftResTime[set] = Time[leftIndex];
            }
            break;
         }
         
         //--- Draw Support & Resistance
         color supColor = (set == 0) ? InpSupColor1 : (set == 1) ? InpSupColor2 : InpSupColor3;
         color resColor = (set == 0) ? InpResColor1 : (set == 1) ? InpResColor2 : InpResColor3;
         
         if(leftSupTime[set] > 0 && rightSupTime[set] > 0) {
            drawLine("Current_Support_" + IntegerToString(set+1), 
                     leftSupTime[set], leftSupPrice[set], 
                     rightSupTime[set], rightSupPrice[set], supColor);
         }
         if(leftResTime[set] > 0 && rightResTime[set] > 0) {
            drawLine("Current_Resistance_" + IntegerToString(set+1), 
                     leftResTime[set], leftResPrice[set], 
                     rightResTime[set], rightResPrice[set], resColor);
         }
      }
   }
   
   // Check alerts on every tick
   CheckAlerts(Close[0]);
   
   return(0);
}
//+------------------------------------------------------------------+
//| Check for price approaching lines                                |
//+------------------------------------------------------------------+
void CheckAlerts(double currentPrice) {
   if(!alertsEnabled) return;
   
   // Prevent alert spam - max 1 alert per bar
   if(lastAlertTime == Time[0]) return;
   
   // Calculate alert distance as percentage of current price
   double alertDistance = currentPrice * InpAlertOffset / 100.0;
   
   for(int set = 0; set < 3; set++) {
      // Check Support
      if(leftSupTime[set] > 0 && rightSupTime[set] > 0) {
         double linePrice = GetLinePrice(leftSupTime[set], leftSupPrice[set], 
                                         rightSupTime[set], rightSupPrice[set], TimeCurrent());
         
         if(currentPrice > linePrice && (currentPrice - linePrice) <= alertDistance) {
            Alert(Symbol(), " ", GetPeriodString(), ": Price approaching Support Set ", set+1, 
                  " | Price: ", DoubleToString(currentPrice, Digits),
                  " | Line: ", DoubleToString(linePrice, Digits));
            lastAlertTime = Time[0];
            return;
         }
      }
      
      // Check Resistance
      if(leftResTime[set] > 0 && rightResTime[set] > 0) {
         double linePrice = GetLinePrice(leftResTime[set], leftResPrice[set], 
                                         rightResTime[set], rightResPrice[set], TimeCurrent());
         
         if(currentPrice < linePrice && (linePrice - currentPrice) <= alertDistance) {
            Alert(Symbol(), " ", GetPeriodString(), ": Price approaching Resistance Set ", set+1,
                  " | Price: ", DoubleToString(currentPrice, Digits),
                  " | Line: ", DoubleToString(linePrice, Digits));
            lastAlertTime = Time[0];
            return;
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Get line price at specific time                                  |
//+------------------------------------------------------------------+
double GetLinePrice(datetime t1, double p1, datetime t2, double p2, datetime targetTime) {
   if(t1 == t2) return p2;
   double slope = (p2 - p1) / (double)(t2 - t1);
   return p2 + slope * (double)(targetTime - t2);
}
//+------------------------------------------------------------------+
//| Convert period to string                                         |
//+------------------------------------------------------------------+
string GetPeriodString() {
   switch(Period()) {
      case PERIOD_M1:  return "M1";
      case PERIOD_M5:  return "M5";
      case PERIOD_M15: return "M15";
      case PERIOD_M30: return "M30";
      case PERIOD_H1:  return "H1";
      case PERIOD_H4:  return "H4";
      case PERIOD_D1:  return "D1";
      case PERIOD_W1:  return "W1";
      case PERIOD_MN1: return "MN";
      default:         return IntegerToString(Period());
   }
}
//+------------------------------------------------------------------+
//| The Local Low search function                                    |
//+------------------------------------------------------------------+
bool isLowestLow(int bar, int side) {
   for(int i = 1; i <= side; i++) {
      if(Low[bar] > Low[bar-i] || Low[bar] > Low[bar+i]) {
         return false;
      }
   }
   return true;
}
//+------------------------------------------------------------------+
//| The Local High search function                                   |
//+------------------------------------------------------------------+
bool isHighestHigh(int bar, int side) {
   for(int i = 1; i <= side; i++) {
      if(High[bar] < High[bar-i] || High[bar] < High[bar+i]) {
         return false;
      }
   }
   return true;
}
//+------------------------------------------------------------------+
//| Draw trend line function                                         |
//+------------------------------------------------------------------+
void drawLine(string name, datetime t1, double p1, datetime t2, double p2, color clr) {
   ObjectDelete(name);
   ObjectCreate(name, OBJ_TREND, 0, t1, p1, t2, p2);
   ObjectSet(name, OBJPROP_COLOR, clr);
   ObjectSet(name, OBJPROP_WIDTH, InpLinesWidth);
   ObjectSet(name, OBJPROP_STYLE, STYLE_SOLID);
   ObjectSet(name, OBJPROP_RAY, true);
   ObjectSet(name, OBJPROP_SELECTABLE, true);
   ObjectSet(name, OBJPROP_BACK, false);
   if(!showLines) ObjectSet(name, OBJPROP_TIMEFRAMES, OBJ_NO_PERIODS);
}
//+------------------------------------------------------------------+
//| Create Button                                                    |
//+------------------------------------------------------------------+
void CreateButton(string name, int x, int y, int width, int height, string text, color txtColor, color bgColor) {
   ObjectCreate(name, OBJ_BUTTON, 0, 0, 0);
   ObjectSet(name, OBJPROP_CORNER, CORNER_RIGHT_LOWER);
   ObjectSet(name, OBJPROP_XDISTANCE, x);
   ObjectSet(name, OBJPROP_YDISTANCE, y);
   ObjectSet(name, OBJPROP_XSIZE, width);
   ObjectSet(name, OBJPROP_YSIZE, height);
   ObjectSetText(name, text, 9, "Arial", txtColor);
   ObjectSet(name, OBJPROP_COLOR, txtColor);
   ObjectSet(name, OBJPROP_BGCOLOR, bgColor);
   ObjectSet(name, OBJPROP_SELECTABLE, false);
}
//+------------------------------------------------------------------+